/** @file   player.cpp
 * @brief   Implementation of Player - class.
 * @version $Revision: 1.9 $
 * @author  Tomi Lamminsaari
 */
 
 
#include "player.h"
#include "gameanims.h"
#include "settings.h"
#include "consts.h"
using namespace eng2d;
#include "bonusobject.h"
#include "animplayer.h"
#include "www_map.h"
#include "utils.h"
#include "GfxManager.h"
#include "warglobals.h"
#include "redrawqueue.h"
#include "door.h"
#include "playercontroller.h"
#include <string>
#include <allegro.h>
#include "www_assert.h"
#include "AnimId.h"
#include "LightBeam.h"
using std::ifstream;
using std::ofstream;
using std::endl;
#include "soundsamples.h"

namespace WeWantWar {


//********************************************************************
//                                                                   *
//      Static members and constants                                 *
//                                                                   *
//********************************************************************

Vec2D Player::forwardVec(0,-4.4);
Vec2D Player::backwardVec(0,3.0);
Vec2D Player::leftVec(-3.0, 0);
Vec2D Player::rightVec(3.0, 0);

Vec2D Player::forwardSpeed( 0,-4.4 );
Vec2D Player::strafeSpeed( 0, -3.3 );
Vec2D Player::backwardSpeed( 0, -3.0 );


const int Player::WEAPON_CHANGE_DELAY = 30;
const int Player::UZI_1_CTRLPOINT;
const int Player::UZI_2_CTRLPOINT;
const int Player::AMMOS_OUT_CTRLPOINT;
const int Player::UZIAMMOS_OUT_CTRLPOINT1;
const int Player::UZIAMMOS_OUT_CTRLPOINT2;
const int Player::ROCKETLAUNCHER_NOSE_CTRLPOINT;
const int Player::ROCKETLAUNCHER_SHOOTFLAME_CTRLPOINT;

const int Player::RELOAD_COUNTER_INDEX;
const int Player::INVULNERABILITY_COUNTER_INDEX;


//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Constructor
 */
Player::Player() :
  GameObject( ),
  m_weapon( Weapon::W_CROWBAR ),
  m_grenades( -1 ),
  m_shield( 0 ),
  m_activeUzi( 0 ),
  m_pVehicle( 0 ),
  m_sniperMode( false ),
  iFlashLight( 0 )
{
  ObjectID::Type oid = ObjectID::TYPE_PLAYER;
  
  int animID = GameAnims::EIdle;
  const Animation& anim = GameAnims::findAnimation( AnimId::KPlayerCrowbar, animID );
  this->setAnimation( anim, animID );
  
  // Set the collisionpoints
  this->setCollisionPoint( 0, Vec2D( -12,-12 ) );
  this->setCollisionPoint( 1, Vec2D(  12,-12 ) );
  this->setCollisionPoint( 2, Vec2D(  12, 12 ) );
  this->setCollisionPoint( 3, Vec2D( -12, 12 ) );
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  if ( Settings::difficultyLevel == Settings::KHard ) {
    this->setArmor( Settings::floatObjProp(oid, "armor2:") );
  } else {
    this->setArmor( Settings::floatObjProp(oid, "armor1:") );
  }
  
  // Set propertyflags.
  this->setProperties( GameObject::PROP_GRENADEFORCE );
  
  
  // Adds the controlpoint for the weapon's pipe
  this->addCtrlPoint( Vec2D( 4, -24) );   // rifle/shotgun/flamethrowe/minigun
  this->addCtrlPoint( Vec2D( 8, -20) );   // uzi 1
  this->addCtrlPoint( Vec2D(-8, -20) );   // uzi 2
  this->addCtrlPoint( Vec2D( 4, -6 ) );   // ammos out
  this->addCtrlPoint( Vec2D( 5, -16) );   // uzi1 ammos out
  this->addCtrlPoint( Vec2D(-5, -16) );   // uzi2 ammos out
  this->addCtrlPoint( Vec2D( 8, -17) );   // rocket launcher's nose
  this->addCtrlPoint( Vec2D( 7,  13) );   // rocket launcher's rear end
  
  // Create the controller
  this->setController( new PlayerController( this ) );
  
  // Create the weapon tables.
  for (int i=0; i < Weapon::WEAPON_COUNT; i++) {
    m_weaponList.push_back( -1 );
  }
  //m_weaponList.at(11) = 10000;
  m_weaponList.at( m_weapon.id() ) = Weapon::maxPlayerAmmo.at( m_weapon.id() );
  
  // create flashlight light beam
  BITMAP* lightbeam = load_bitmap( "gfx/lights/FlashLightBeam.bmp", 0 );
  if ( lightbeam != 0 ) {
    iFlashLight = new LightBeam( Settings::lightbeamMatrixCellSize );
    iFlashLight->constructFromBitmap( lightbeam );
    destroy_bitmap( lightbeam );
  }
}



/** Destructor
 */
Player::~Player()
{
  delete iFlashLight;
}



//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

/** The update method
 */
void Player::update()
{
  if ( this->state() == GameObject::STATE_KILLED ||
       this->state() == GameObject::STATE_HIBERNATING ) {
    return;
  }
  
  
  iFlashLight->setBeamPosition( this->position(), this->angle() );
  iFlashLight->update();
  
  // Throwing the grenade is implemented so, that when player presses the
  // "throw grenade"-key, we don't spawn the grenade but set the throwing
  // animation. When the throwing animation has been shown through, we
  // spawn new grenade to the "global" grenade table.
  if ( this->getAnimID() == GameAnims::EThrowing && m_animation.paused() ) {
    // The throw-grenade animation has been played through. Throw
    // grenade.
    this->throwGrenade();
    
    // Set the default animation back.
    const Animation& a = GameAnims::getAnimByWeapon(m_weapon.id(), GameAnims::EWalk );
    this->setAnimation(a, GameAnims::EWalk);
  }
  // If the hitting with crowbar animation has stopped running, we set back
  // the default walking animation.
  if ( m_weapon.id() == Weapon::W_CROWBAR && m_animation.paused() ) {
    const Animation& a = GameAnims::getAnimByWeapon( m_weapon.id(), GameAnims::EWalk );
    this->setAnimation( a, GameAnims::EWalk );
  }
  
  // Read the keyboard and mouse and do movements.
  if ( Settings::controlMode == Settings::CONTROLMODE1 ) {
    handleControls();
  } else if ( Settings::controlMode == Settings::CONTROLMODE2 ) {
    handleControls2();
  }
  
  // If we did touch a wall during this update-round, we check if
  // we should open a door.
  if ( this->getWallTouch() == true ) {
    this->checkDoors();
  }
  
  if ( this->getCounter( NOENTER_COUNTER_INDEX ) < 0 ) {
    if ( key[KEY_E] ) {
      WarGlobals::captureVehicle = true;
    }
  }
}



/** Draws this object.
 */
void Player::redraw( RedrawQueue* pQueue )
{
  if ( this->hidden() == true ) {
    return;
  }
  
  Vec2D p = this->position();
  int plaX = static_cast<int>( p.x() ) - Map::scrollX;
  int plaY = static_cast<int>( p.y() ) - Map::scrollY;

  BITMAP* pS = m_animation.currentFrame().asBitmap();
  plaX -= pS->w / 2;
  plaY -= pS->h / 2;

  // If the player is invulnerable he blinks.
  int invc = this->getCounter( INVULNERABILITY_COUNTER_INDEX );
  if ( invc >= 0 ) {
    if ( invc % 2 == 0 ) {
      return;
    }
  }
  
  pQueue->addRotatedSprite( RedrawQueue::PRI_ABOVE_NORMAL, plaX,plaY,
                            itofix( m_angle ), pS );
}



/** We got hit by a bullet
 */
bool Player::hitByBullet( Bullet* pB )
{
  if ( this->state() == GameObject::STATE_HIBERNATING ) {
    return false;
  }
  
  if ( this->getCounter( INVULNERABILITY_COUNTER_INDEX ) >= 0 ) {
    return false;
  }


  // If this is a grenade fragment, it causes as some damage
  if ( pB->iType == Bullet::EGrenade ) {
    this->causeDamage( pB );
    ParticleBlood* pP = new ParticleBlood( pB->iPosition,
                                           pB->velocity(), 12,
                                           Color(130,60,10) );
    WarGlobals::pPartManager->addSystem( pP );
    return true;
  }
  
  // This was shot by player so it does not affect us.
  if ( pB->iOwner == this ) {
    return false;
  }
  // Also the bullets shot by NpcCharacter - object do not hit us.
  if ( pB->iOwner != 0 ) {
    if ( pB->iOwner->objectType() == ObjectID::TYPE_SIMON ||
         pB->iOwner->objectType() == ObjectID::TYPE_ERIC ||
         pB->iOwner->objectType() == ObjectID::TYPE_YOUKO ) {
      return false;
    }
  }
  
  // Any other bullets causes us damage.
  this->causeDamage( pB );
  if ( m_shield > 0 ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KForceField );
    AnimPlayer::spawn( anim, this->position() );
  }
  
  if ( pB->iType != Bullet::EFlameThrower ) {
    ParticleBlood* pP = new ParticleBlood( pB->iPosition, pB->velocity(), 12,
                                           Color(130,60,10) );
    WarGlobals::pPartManager->addSystem( pP );
  }
  return true;
}



/** Causes damage
 */
bool Player::causeDamage( int points )
{
  if ( this->state() == GameObject::STATE_HIBERNATING ) {
    return false;
  }
  
  if ( this->getCounter( INVULNERABILITY_COUNTER_INDEX ) < 0 ) {
    if ( m_shield < 0 ) {
      return GameObject::causeDamage( points );
    } else {
      m_shield -= static_cast<float>( points );
    }
  }
  return false;
}



/** Causes damage
 */
bool Player::causeDamage( Bullet* b )
{
  if ( this->state() == GameObject::STATE_HIBERNATING ) {
    return false;
  }
  m_shield -= b->iDamage;
  if ( m_shield < 0 ) {
    if ( this->getCounter( INVULNERABILITY_COUNTER_INDEX ) < 0 ) {
      return GameObject::causeDamage( b );
    }
  }
  return false;
}


/** Kills this player.
 */
void Player::kill()
{
  if ( this->state() == GameObject::STATE_KILLED ||
       this->state() == GameObject::STATE_DYING ) {
    return;
  }
  this->state( STATE_KILLED );
}



/** Plays player's samples.
 */
void Player::makeSound( GameObject::SoundID id ) const
{
  switch (id) {
    case ( GameObject::SND_PAIN ): {
      Sound::playSample( SMP_PLAYER_PAIN, false );
      break;
    }
    case ( GameObject::SND_ATTACK ): {
      Sound::playSample( m_weapon.getSoundID(), false );
      break;
    }
    case ( GameObject::SND_RELOAD ): {
      if ( m_weapon.id() == Weapon::W_SNIPERRIFLE ) {
        Sound::playSample( SMP_NEWWEAPON, false );
      }
      break;
    }
    default: {
      break;
    }
  }
}



/** Picks up the bonus.
 */
bool Player::pickupBonus( BonusObject* b )
{
  if ( this->state() == GameObject::STATE_HIBERNATING ) {
    return false;
  }
  
  // If the provided BonusObject is a weapon, we first check does player
  // have that weapon already. If he doesn't we start using that weapon but
  // if he already has that weapon, we only add the ammunition count unless
  // the bulletcount is full.
  switch ( b->type() ) {

    // Handle the RIFLE-bonus item
    case ( BonusObject::RIFLE ): {
      return this->pickWeapon( Weapon::W_RIFLE );
    }
    
    // Handle the FIRST AID KIT bonus item
    case ( BonusObject::FIRSTAID ): {
      if ( m_health >= 100 ) {
        return false;
      }
      m_health += Settings::firstaid_points;
      if ( m_health > 100 ) {
        m_health = 100;
      }
      Sound::playSample( SMP_FIRSTAID, false );
      return true;
    }
    
    // Handle the SHOTGUN-bonus item
    case ( BonusObject::SHOTGUN ): {
      return this->pickWeapon( Weapon::W_SHOTGUN );
    }
    
    // Handle the GRENADEPACK-bonus item
    case ( BonusObject::GRENADE ): {
      if ( m_grenades >= Weapon::MAX_GRENADES ) {
        return false;
      }
      m_grenades += Weapon::GRENADE_PACK_SIZE;
      Sound::playSample( SMP_NEWWEAPON, false );
      return true;
    }
    
    // Handle the FLAMETHROWER-bonus item
    case ( BonusObject::FLAMETHROWER ): {
      return this->pickWeapon( Weapon::W_FLAMETHROWER );
    }
    
    case ( BonusObject::MINIGUN ): {
      return this->pickWeapon( Weapon::W_MINIGUN );
    }
    
    case ( BonusObject::UZI ): {
      return this->pickWeapon( Weapon::W_UZI );
    }
    
    case ( BonusObject::EXTRALIFE ): {
      WarGlobals::numberOfLives += 1;
      Sound::playSample( SMP_EXTRALIFE, false );
      return true;
    }
    case ( BonusObject::SCORE_100PTS ): {
      WarGlobals::gameStats.grantScores( 100 );
      Sound::playSample( SMP_NEWWEAPON, false );
      return true;
    }
    case ( BonusObject::SCORE_200PTS ): {
      WarGlobals::gameStats.grantScores( 200 );
      Sound::playSample( SMP_NEWWEAPON, false );
      return true;
    }
    case ( BonusObject::SNIPERRIFLE ): {
      return this->pickWeapon( Weapon::W_SNIPERRIFLE );
    }
    case ( BonusObject::ROCKETLAUNCHER ): {
      return this->pickWeapon( Weapon::W_ROCKETLAUNCHER );
    }
    case ( BonusObject::BODYARMOR ): {
      if ( m_shield < Settings::floatObjProp(ObjectID::TYPE_PLAYER, "max_shield:") ) {
        m_shield = Settings::floatObjProp(ObjectID::TYPE_PLAYER, "shield_pack:");
        Sound::playSample(SMP_FIRSTAID, false);
        return true;
      }
      break;
    }
  }
  return false;
}



/** Picks up the ammunition.
 */
bool Player::pickWeapon(int w)
{
  if ( m_weaponList.at( w ) >= Weapon::getMaxAmmo( w ) ) {
    // The weapon is full of ammos
    return false;
  }
  
  if ( m_weaponList.at( w ) == -1 ) {
    // We didn't have this weapon yet, so we change to it.
    m_weaponList.at( w ) = Weapon::getAmmoPack( w );
    this->changeWeapon( w );
    // And we request the mainprogram to show weaponguide
    this->requestWeaponguide( w );
    
   } else {
    // We had this weapon already, so we just add the ammunition count
    m_weaponList.at( w ) += Weapon::getAmmoPack( w );
    Sound::playSample( SMP_NEWWEAPON, false );
  }
  return true;
}



/** Saves the player's data to given stream
 */
void Player::savePlayerData( ofstream& fout ) const
{
  fout << "[PLAYER]" << endl;
  fout << "health: " << this->health() << endl;
  fout << "armor: " << this->armor() << endl;
  fout << "current_weapon: " << this->getWeapon() << endl;
  fout << "ammunition: " << endl;
  for (int i=0; i < m_weaponList.size(); i++) {
    fout << m_weaponList.at(i) << " ";
  }
  fout << endl;
  
  fout << "grenades: " << this->getGrenadeCount() << endl;
  fout << "[END_OF_PLAYER]" << endl;
}



/** Loads the player's data from given stream
 */
void Player::loadPlayerData( ifstream& fin )
{
  std::string tmp;
  fin >> tmp;
  if ( tmp != "[PLAYER]" ) {
    return;
  }
  
  int wid;
  fin >> tmp >> GameObject::m_health;
  fin >> tmp >> GameObject::m_armor;
  fin >> tmp >> wid;
  m_weapon = Weapon( wid );
  fin >> tmp;
  for (int i=0; i < m_weaponList.size(); i++) {
    int amount;
    fin >> amount;
    if ( Settings::cheatAmmo == false ) {
      m_weaponList.at(i) = amount;
    }
  }
  fin >> tmp >> m_grenades;
  fin >> tmp;
}




/** Wakes up the dead player
 */
void Player::resurrect()
{
  this->state( GameObject::STATE_LIVING );
  this->setHealth( 100 );
  
  int animID = GameAnims::EIdle;
  const Animation& rA = GameAnims::getAnimByWeapon( m_weapon.id(), animID );
  this->setAnimation( rA, animID );
  
  this->setCounter( INVULNERABILITY_COUNTER_INDEX, 80 );
  
  if ( m_pVehicle != 0 ) {
    m_pVehicle->resurrect();
    this->hidden( true );
    this->state( GameObject::STATE_HIBERNATING );
  }
}



/** Connects us with given object
 */
void Player::setVehicle( GameObject* pObj )
{
  m_pVehicle = pObj;
  
  if ( pObj == 0 ) {
    this->setCounter( NOENTER_COUNTER_INDEX, 80 );
  }
}




//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************

/** Tells if the player's weapon is still in reloadin mode.
 */
bool Player::reloading() const
{
  if ( this->getCounter( RELOAD_COUNTER_INDEX ) < 0 ) {
    return false;
  }
  return true;
}



/** Returns the type of player's current weapon
 */
int Player::getWeapon() const
{
  return m_weapon.id();
}



/** Returns the number of ammunition the current weapon has.
 */
int Player::getAmmunition() const
{
  return m_weaponList.at( m_weapon.id() );
}



/** Tells the grenade count
 */
int Player::getGrenadeCount() const
{
  return m_grenades;
}



/** Returns the type of this object.
 */
ObjectID::Type Player::objectType() const
{
  return ObjectID::TYPE_PLAYER;
}



/** Returns our vehicle
 */
GameObject* Player::getVehicle() const
{
  return m_pVehicle;
}



/** Tells the sniper mode status
 */
bool Player::sniperMode() const
{
  return m_sniperMode;
}


LightBeam* Player::FlashLightBeam() const
{
  return iFlashLight;
}



//********************************************************************
//                                                                   *
//      Private methods                                              *
//                                                                   *
//********************************************************************

/** Requests the weaponguide
 */
void Player::requestWeaponguide( int wid )
{

}

/** Handles the controls and moves the player by that
 */
void Player::handleControls()
{
  PlayerController* pC = dynamic_cast<PlayerController*>( this->getController() );
  pC->update();
  
  if ( WarGlobals::bulletTimeMode == true ) {
    FpsTimer* timerHandler = FpsTimer::getInstance();
    unsigned int updatecount = timerHandler->logicalUpdatesCount();
    FpsTimer::releaseInstance( timerHandler );
    
    if ( updatecount % 2 == 0 ) {
      // These are updated only on every even tick since every odd tick we
      // update the whole object.
      this->changeAngle( pC->turn() );
      m_counters[RELOAD_COUNTER_INDEX] -= 1;
      if ( m_weapon.id() == Weapon::W_CROWBAR &&
           this->getAnimID() == GameAnims::EShoot ) {
        m_animation.update();
      }
      return;
    }
  }
  
  // If we're in sniper mode, the player is controlled differently.
  if ( m_sniperMode == true ) {
    this->handleSniperModeControls( pC );
    return;
  }
  
  if ( pC->forward() != 0 ) {
    if ( pC->left() != 0 ) {
      // Move forward and strafe left
      Vec2D v( forwardVec );
      v.rotate( 224 );
      v.rotate( this->angle() );
      this->move( v );
      
    } else if ( pC->right() != 0 ) {
      Vec2D v( forwardVec );
      v.rotate( 32 );
      v.rotate( this->angle() );
      this->move( v );
      
    } else {
      Vec2D v( forwardVec );
      v.rotate( this->angle() );
      this->move( v );
      
    }

  } else if ( pC->backward() != 0 ) {
    if ( pC->left() != 0 ) {
      Vec2D v( backwardVec );
      v.rotate( 32 );
      v.rotate( this->angle() );
      this->move( v );
      
    } else if ( pC->right() != 0 ) {
      Vec2D v( backwardVec );
      v.rotate( 224 );
      v.rotate( this->angle() );
      this->move( v );
      
    } else {
      Vec2D v( backwardVec );
      v.rotate( this->angle() );
      this->move( v );
      
    }

  } else if ( pC->left() != 0 ) {
    Vec2D v( strafeSpeed );
    v.rotate( this->angle() + 192 );
    this->move( v );
  } else if ( pC->right() != 0 ) {
    Vec2D v( strafeSpeed );
    v.rotate( this->angle() + 64 );
    this->move( v );
    
  }
  
  
  // Turn the player
  this->changeAngle( pC->turn() );
  
  
  // Handle the weapon change
  if ( pC->weaponChange() != 0 ) {
    this->changeWeapon( this->getSelectedWeaponId( pC->weaponChange() ) );
  }
  
  // If player is shooting, handle it. This needs to be done before you
  // call manageAnimation()-method.
  if ( pC->shoot() != 0 ) {
    this->handleShooting();
  }
  
  // Choose the correct animation based on current weapon and the movements.
  this->manageAnimations();
  
  // Remote detonate the grenades
  if ( pC->detonate() != 0 ) {
    WarGlobals::pBulletManager->detonateGrenades( this );
  }
  
  // Throw grenade. This is done by setting up the throwing animation.
  // In the update()-method we check when the throwing animation has been
  // shown and spawn the grenade then.
  if ( pC->grenade() && m_grenades > 0 ) {
    if ( this->getAnimID() != GameAnims::EThrowing ) {
      int t = GameAnims::EThrowing;
      const Animation& anim = GameAnims::getAnimByWeapon( m_weapon.id(), t );
      this->setAnimation( anim, t );
    }
  }
  
  // Set the mousecursor's position.
  Vec2D distV( 0, Consts::CROSSHAIR_DISTANCE );
  distV.rotate( this->angle() );
  distV += this->position();
  distV -= Vec2D( Map::scrollX, Map::scrollY );
  WarGlobals::pHud->crosshairPos( distV );
}



/** Handles the controls when using the controlmode 2
 */
void Player::handleControls2()
{
  // Handle the rotation by mouse
  PlayerController* pC = dynamic_cast<PlayerController*>( this->getController() );
  pC->update();
  
  if ( WarGlobals::bulletTimeMode == true ) {
    FpsTimer* timerHandler = FpsTimer::getInstance();
    unsigned int updatecount = timerHandler->logicalUpdatesCount();
    FpsTimer::releaseInstance( timerHandler );
    
    if ( updatecount % 2 == 0 ) {
      m_counters[ RELOAD_COUNTER_INDEX ] -= 1;
      Vec2D mvec( pC->sightX(), pC->sightY() );
      fixed a = Utils::findVecAngle( this->position(), mvec );
      this->angle( fixtoi( a ) );
      return;
    }
  }
  
  // If we're in sniper mode, the controls are handled differently
  if ( m_sniperMode == true ) {
    this->handleSniperModeControls( pC );
    return;
  }
  
  Vec2D mvec( pC->sightX(), pC->sightY() );
  fixed a = Utils::findVecAngle( this->position(), mvec );
  this->angle( fixtoi( a ) );
  
  
  // If we're shooting, we handle it. This needs to be done before we
  // manage the animations.
  if ( pC->shoot() != 0 ) {
    this->handleShooting();
  }
  
  if ( pC->forward() && !pC->left() && !pC->right() ) {
    // Move up
    this->moveUp();

  } else if ( pC->forward() && pC->left() && !pC->right() ) {
    // Move up left
    this->moveUpLeft();

  } else if ( pC->forward() && !pC->left() && pC->right() ) {
    // move up right
    this->moveUpRight();

  } else if ( pC->backward() && !pC->left() && !pC->right() ) {
    // Move down
    this->moveDown();

  } else if ( pC->backward() && pC->left() && !pC->right() ) {
    // move down left
    this->moveDownLeft();

  } else if ( pC->backward() && !pC->left() && pC->right() ) {
    // move down right
    this->moveDownRight();

  } else if ( pC->left() && !pC->right() ) {
    // move left
    this->moveLeft();

  } else if ( !pC->left() && pC->right() ) {
    this->moveRight();

  } else {
    // We're not moving
    int animID = GameAnims::EIdle;
    if ( pC->shoot() ) animID = GameAnims::EShoot;
    if ( this->getAnimID() != animID && this->getAnimID() != GameAnims::EThrowing ) {
      const Animation& a = GameAnims::getAnimByWeapon(m_weapon.id(), animID);
      this->setAnimation( a, animID );
    }
  }

  
  if ( pC->weaponChange() != 0 ) {
    this->changeWeapon( this->getSelectedWeaponId(pC->weaponChange() ) );
    const Animation& an = GameAnims::getAnimByWeapon(m_weapon.id(), GameAnims::EIdle);
    this->setAnimation( an, GameAnims::EIdle );
  }
  
  // Remote detonate the grenades
  if ( pC->detonate() ) {
    WarGlobals::pBulletManager->detonateGrenades( this );
  }
  
  
  // Throw grenade. This is done by setting up the throwing animation.
  // In the update()-method we check when the throwing animation has been
  // shown and spawn the grenade then.
  if ( pC->grenade() && m_grenades > 0 ) {
    if ( this->getAnimID() != GameAnims::EThrowing ) {
      int t = GameAnims::EThrowing;
      const Animation& anim = GameAnims::getAnimByWeapon( m_weapon.id(), t );
      this->setAnimation( anim, t );
    }
  }
  
  // Set mousecursor's position.
  Vec2D cursorpos( pC->sightX() - Map::scrollX, pC->sightY() - Map::scrollY );
  WarGlobals::pHud->crosshairPos( cursorpos );
}



/** Handles the sniper mode controls
 */
void Player::handleSniperModeControls( PlayerController* aController )
{
  // Find the angle where we are looking at.
  Vec2D mvec( aController->sightX(), aController->sightY() );
  fixed a = Utils::findVecAngle( this->position(), mvec );
  this->angle( fixtoi( a ) );
  
  
  // Change weapon if necessary
  if ( aController->weaponChange() != 0 ) {
    Weapon::WeaponID weaponId =
        this->getSelectedWeaponId( aController->weaponChange() );
    this->changeWeapon( weaponId );
    if ( m_weapon.id() != Weapon::W_SNIPERRIFLE ) {
      // We are no longer using the sniper rifle.
      return;
    }
  }

  // In sniper mode we don't have any animation.
  const Animation& anim = GameAnims::getAnimByWeapon( m_weapon.id(),
                                                      GameAnims::EShoot );
  this->setAnimation( anim, GameAnims::EShoot );
  
  // If player moves, we set the sniper mode off.
  if ( aController->forward() != 0 || aController->backward() != 0 ||
       aController->left() != 0 || aController->right() != 0 ) {
    m_sniperMode = false;
  }
  
  // Check if player wants to shoot
  if ( aController->shoot() != 0 ) {
    m_weapon = Weapon( Weapon::W_SNIPERRIFLE );
    this->handleShooting();
  }
  
  // Set mousecursor position
  Vec2D cursorpos( aController->sightX() - Map::scrollX,
                   aController->sightY() - Map::scrollY );
  WarGlobals::pHud->crosshairPos( cursorpos );
}



/** Handles the shooting with current weapon.
 */
void Player::handleShooting()
{
  // We do not shoot while we're throwing a grenade
  if ( this->getAnimID() == GameAnims::EThrowing ) {
    return;
  }
  
  // Are we still reloading our current weapon
  if ( this->reloading() ) {
    return;
  }
  
  // Check if current weapon is out of ammos
  if ( m_weaponList.at(m_weapon.id()) <= 0 ) {
    // No ammos. We try to change the weapon.
    if ( m_weaponList.at( Weapon::W_SHOTGUN ) > 0 ) {
      this->changeWeapon( Weapon::W_SHOTGUN );
      
    } else if ( m_weaponList.at( Weapon::W_RIFLE ) > 0 ) {
      this->changeWeapon( Weapon::W_RIFLE );
      
    } else if ( m_weaponList.at( Weapon::W_UZI ) > 0 ) {
      this->changeWeapon( Weapon::W_UZI );
      
    } else if ( m_weaponList.at( Weapon::W_MINIGUN ) > 0 ) {
      this->changeWeapon( Weapon::W_MINIGUN );
      
    } else {
      this->changeWeapon( Weapon::W_CROWBAR );
      
    }
    return;
  }
  
  // Reduce the bullet count and spawn new bullet to the BulletStorage
  m_weaponList.at( m_weapon.id() ) -= 1;
  
  Weapon* pW = 0;
  switch ( m_weapon.id() ) {
    case ( Weapon::W_RIFLE ): {
      this->shootRiffle();
      break;
    }
    case( Weapon::W_SHOTGUN ): {
      this->shootShotgun();
      break;
    }
    case ( Weapon::W_CROWBAR ): {
      this->hitHammer();
      break;
    }
    case ( Weapon::W_FLAMETHROWER ): {
      this->shootFlamethrower();
      break;
    }
    case ( Weapon::W_MINIGUN ): {
      this->shootMinigun();
      break;
    }
    case ( Weapon::W_UZI ): {
      this->shootUzi();
      break;
    }
    case ( Weapon::W_SNIPERRIFLE ): {
      this->shootSniperRifle();
      break;
    }
    case ( Weapon::W_ROCKETLAUNCHER ): {
      this->shootRocketLauncher();
      break;
    }
    default: {
      textprintf_ex(screen,font,0,0,makecol(255,255,255),0, "%d", m_weapon.id() );
      alert( "Unknown weapon error", 0, 0, "ok",0, 0,0 );
      break;
    }
  }
}



/** Shoots the riffle
 */
void Player::shootRiffle()
{
  Weapon::Specs& rW = Weapon::weaponlist.at( Weapon::W_RIFLE );
  
  
  // Calculate the speed vector
  Vec2D spdVec( 0, rW.speed );
  spdVec.rotate( this->angle() );

  // Calculate the coordinate of the gun's pipe
  Vec2D gunV = this->position() + this->getCtrlPoint( RIFLE_CTRLPOINT );


  // Create new bullet and add it to the bullet table
  Bullet* pB = BulletTable::createBullet( this, gunV,
                                          Bullet::ERifle );
  WarGlobals::pBulletManager->spawnBullet( pB );

  // Set the flame and light animations. Make sound
  ParticleSystem* particles = new ParticleGunFlames( gunV, pB->velocity(), 15, 60 );
  WarGlobals::pPartManager->addSystem( particles );
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KRifleShootFlame );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  AnimPlayer::spawn( flameAnim, gunV, 0 );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV, 0 );
  }

  this->makeSound( GameObject::SND_ATTACK );
  
  // Show dropping bullet animation.
  Vec2D ammoPos( this->getCtrlPoint( AMMOS_OUT_CTRLPOINT ) );
  ammoPos += this->position();
  Vec2D ammoDir(0, -3.8);
  int v = (rand() % 10) - 5;
  ammoDir.rotate( this->angle() + 64 + v );
  const Animation& bulletShellAnim =
      GameAnims::findAnimation( AnimId::KEffectBulletShell );
  AnimPlayer::spawn( bulletShellAnim, ammoPos, ammoDir, AnimPlayer::EBulletShell );
  
  this->setCounter( RELOAD_COUNTER_INDEX, rW.reload );
  
  
  // Update the statistics
  WarGlobals::gameStats.updateStatistics( Statistics::SHOTS, 0 );
}



/** Shoots the shotgun
 */
void Player::shootShotgun()
{
  Weapon::Specs& rW = Weapon::weaponlist.at( Weapon::W_SHOTGUN );

  // Shotgun releases 6 bullets
  Vec2D gunV = this->position() + this->getCtrlPoint( RIFLE_CTRLPOINT );
  
  for (int i=0; i < 6; i++) {
    Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::EShotgun );
    WarGlobals::pBulletManager->spawnBullet( pB );
    
    // We update the statistics here since each bullet must update the
    // statistics
    WarGlobals::gameStats.updateStatistics( Statistics::SHOTS, 0 );
  }

  // Set the flame and light animations. Make sound
  Vec2D dirV( 0, -16 );
  dirV.rotate( this->angle() );
  ParticleSystem* particles = new ParticleGunFlames( gunV, dirV, 15, 60 );
  WarGlobals::pPartManager->addSystem( particles );
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KShotgunShootFlame );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  AnimPlayer::spawn( flameAnim, gunV, 0 );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV, AnimPlayer::DMODE_ADD,
                       30, Vec2D() );
  }
  
  this->makeSound( GameObject::SND_ATTACK );
  
  // Show dropping bullet animation.
  Vec2D ammoPos( this->getCtrlPoint( AMMOS_OUT_CTRLPOINT ) );
  ammoPos += this->position();
  Vec2D ammoDir(0, -3.8);
  int v = (rand() % 10) - 5;
  ammoDir.rotate( this->angle() + 64 + v );
  const Animation& bulletShellAnim = GameAnims::findAnimation( AnimId::KEffectBulletShell );
  AnimPlayer::spawn( bulletShellAnim, ammoPos, ammoDir, AnimPlayer::EBulletShell );
  this->setCounter( RELOAD_COUNTER_INDEX, rW.reload );
}



/** Hits with the crowbar.
 */
void Player::hitHammer()
{
  // If we're currently showing the hitting animation, we cannot hit again.
  switch ( this->getAnimID() ) {
    case ( GameAnims::EShoot ):
    case ( GameAnims::EShootWalk ):
    case ( GameAnims::EShootStrafe ): {
      return;
    }
  }
  Weapon::Specs& rW = Weapon::weaponlist.at( Weapon::W_CROWBAR );
  
  int angles[] = { 0, -32, 32 };
  
  for (int i=0; i < 3; i++) {
    Vec2D hamV( 0, rW.speed );
    hamV.rotate( this->angle() + angles[i] );
    
    Bullet* pB = BulletTable::createBullet( this, m_position, Bullet::ECrowbar );
    pB->setVelocity( hamV );
    WarGlobals::pBulletManager->spawnBullet( pB );
  }
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( RELOAD_COUNTER_INDEX, rW.reload );
}



/** Shoots with flamethrower
 */
void Player::shootFlamethrower()
{
  Weapon::Specs& rW = Weapon::getWeaponSpecs( Weapon::W_FLAMETHROWER );
  Vec2D gunV = this->position() + this->getCtrlPoint( RIFLE_CTRLPOINT );
  
  Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::EFlameThrower );
  WarGlobals::pBulletManager->spawnBullet( pB );
  
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( RELOAD_COUNTER_INDEX, rW.reload );
  
  // Flamethrower does not increase the shotcount.
}



/** Shoots with minigun
 */
void Player::shootMinigun()
{
  Weapon::Specs& rW = Weapon::getWeaponSpecs( Weapon::W_MINIGUN );
  Vec2D gunV = this->position() + this->getCtrlPoint( RIFLE_CTRLPOINT );
  
  Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::EMinigun );
  WarGlobals::pBulletManager->spawnBullet( pB );
  
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( RELOAD_COUNTER_INDEX, rW.reload );
  
  ParticleSystem* particles = new ParticleGunFlames( gunV, pB->velocity(), 15, 60 );
  WarGlobals::pPartManager->addSystem( particles );
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KMinigunShootFlame );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  AnimPlayer::spawn( flameAnim, gunV, 0 );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV, 0 );
  }
  
  // Show dropping bullet animation.
  Vec2D ammoPos( this->getCtrlPoint( AMMOS_OUT_CTRLPOINT ) );
  ammoPos += this->position();
  Vec2D ammoDir(0, -4.2);
  int v = (rand() % 10) - 5;
  ammoDir.rotate( this->angle() + 64 + v );
  const Animation& shellAnim = GameAnims::findAnimation( AnimId::KEffectBulletShell );
  AnimPlayer::spawn( shellAnim, ammoPos, ammoDir, AnimPlayer::EBulletShell );
  
  // Update the statistics
  WarGlobals::gameStats.updateStatistics( Statistics::SHOTS, 0 );
}



/** Shoots with Uzi
 */
void Player::shootUzi()
{
  Weapon::Specs& rW = Weapon::getWeaponSpecs( Weapon::W_UZI );
  int gunCtrlPoint = ( m_activeUzi == 0 ) ? UZI_1_CTRLPOINT : UZI_2_CTRLPOINT;
  int shellCtrlPoint = ( m_activeUzi == 0 ) ? UZIAMMOS_OUT_CTRLPOINT1 :
                                              UZIAMMOS_OUT_CTRLPOINT2;
  if ( m_activeUzi == 0 ) {
    m_activeUzi = 1;
  } else {
    m_activeUzi = 0;
  }
  
  Vec2D gunV( this->getCtrlPoint( gunCtrlPoint ) );
  gunV += this->position();
  
  Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::EUzi );
  WarGlobals::pBulletManager->spawnBullet( pB );
  
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( RELOAD_COUNTER_INDEX, rW.reload );
  
  ParticleSystem* particles = new ParticleGunFlames( gunV, pB->velocity(), 15, 60 );
  WarGlobals::pPartManager->addSystem( particles );
  
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KUziShootFlame );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV, 0 );
  }
  AnimPlayer::spawn( flameAnim, gunV, 0 );
  
  // Show dropping bullet animation.
  Vec2D ammoPos( this->getCtrlPoint( shellCtrlPoint ) );
  ammoPos += this->position();
    
  Vec2D ammoDir(0, -4.0);
  int v = (rand() % 10) - 5;
  ammoDir.rotate( this->angle() + 64 + v );
  const Animation& shellAnim = GameAnims::findAnimation( AnimId::KEffectBulletShell );
  AnimPlayer::spawn( shellAnim, ammoPos, ammoDir, AnimPlayer::EBulletShell );
  
  // Update the statistics
  WarGlobals::gameStats.updateStatistics( Statistics::SHOTS, 0 );
}



/** Handles the sniper rifle shooting
 */
void Player::shootSniperRifle()
{
  // If we're not in sniper mode, we don't shoot yet. We just go to sniper
  // mode. If we're already in sniper mode, then we shoot.
  if ( m_sniperMode == false ) {
    m_sniperMode = true;
    this->setCounter( RELOAD_COUNTER_INDEX, 30 );
    this->makeSound( GameObject::SND_RELOAD );
    
    // Set mouse position
    Vec2D halfSize( Display::scrWidth()/2, Display::scrHeight()/2 );
    Vec2D mousePos( WarGlobals::pHud->m_cursorpos );
    mousePos -= halfSize;
    mousePos /= Consts::SNIPERMODE_RANGE_FACTOR;
    mousePos += halfSize;
    position_mouse( mousePos.intX(), mousePos.intY() );
    
    // Since the handleShooting() - method reduces the bullet count, we must
    // add sniper bullet. Otherwise just going to sniper mode would take
    // one bullet.
    m_weaponList.at( m_weapon.id() ) += 1;
    return;
  }
  // Calculate the coordinate of the gun's pipe
  Vec2D gunV = this->position() + this->getCtrlPoint( RIFLE_CTRLPOINT );

  // Create new bullet and add it to the bullet table
  Bullet* pB = BulletTable::createBullet( this, gunV,
                                          Bullet::ESniperRifle );
                                          
  // The sniper bullet goes immediately to the target position.
  WarGlobals::pBulletManager->spawnBullet( pB );

  // Set the flame and light animations. Make sound
  ParticleSystem* particles = new ParticleGunFlames( gunV, pB->velocity(), 15, 60 );
  WarGlobals::pPartManager->addSystem( particles );
  const Animation& flameAnim = GameAnims::findAnimation( AnimId::KRifleShootFlame );
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  AnimPlayer::spawn( flameAnim, gunV, 0 );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV, 0 );
  }

  this->makeSound( GameObject::SND_ATTACK );

  // Show dropping bullet animation.
  Vec2D ammoPos( this->getCtrlPoint( AMMOS_OUT_CTRLPOINT ) );
  ammoPos += this->position();
  Vec2D ammoDir(0, -3.8);
  int v = (rand() % 10) - 5;
  ammoDir.rotate( this->angle() + 64 + v );
  const Animation& shellAnim = GameAnims::findAnimation( AnimId::KEffectBulletShell );
  AnimPlayer::spawn( shellAnim, ammoPos, ammoDir, AnimPlayer::EBulletShell );

  Weapon::Specs& rW = Weapon::weaponlist.at( Weapon::W_SNIPERRIFLE );
  this->setCounter( RELOAD_COUNTER_INDEX, rW.reload );

  // Update the statistics
  WarGlobals::gameStats.updateStatistics( Statistics::SHOTS, 0 );
}


/** Shoots with rocket launcher
 */
void Player::shootRocketLauncher()
{
  Weapon::Specs& rW = Weapon::getWeaponSpecs( Weapon::W_ROCKETLAUNCHER );
  Vec2D gunV = this->position() + this->getCtrlPoint( ROCKETLAUNCHER_NOSE_CTRLPOINT );
  Vec2D rearEnd = this->position() +
                  this->getCtrlPoint(ROCKETLAUNCHER_SHOOTFLAME_CTRLPOINT);
  
  Bullet* pB = BulletTable::createBullet( this, gunV, Bullet::ERocketLauncher );
  WarGlobals::pBulletManager->spawnBullet( pB );
  
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( RELOAD_COUNTER_INDEX, rW.reload );
  
  // Show dropping bullet animation.
  Vec2D ammoPos( this->getCtrlPoint( AMMOS_OUT_CTRLPOINT ) );
  ammoPos += this->position();
  Vec2D ammoDir(0, -4.2);
  int v = (rand() % 10) - 5;
  ammoDir.rotate( this->angle() + 64 + v );
  const Animation& shellAnim = GameAnims::findAnimation( AnimId::KEffectBulletShell );
  
  // Update the statistics
  WarGlobals::gameStats.updateStatistics( Statistics::SHOTS, 0 );
}



/** Spawns new grenade
 */
void Player::throwGrenade()
{
  Grenade* g = BulletTable::createGrenade( this, this->position(),
                                           Grenade::EDetonatable );
  WarGlobals::pBulletManager->spawnGrenade( g );
  m_grenades--;
}



/** Manages the animations.
 */
void Player::manageAnimations()
{
  // Default animation is the idle-animation
  PlayerController* pC = dynamic_cast<PlayerController*>( this->getController() );
  int requested = GameAnims::EIdle;
  int current = this->getAnimID();
  
  // If we're running the throwing animation, we let it run through.
  if ( current == GameAnims::EThrowing ) {
    return;
  }
  
  // If we're hitting with crowbar, we let that animation run through.
  if ( m_weapon.id() == Weapon::W_CROWBAR ) {
    if ( current == GameAnims::EShoot || current == GameAnims::EShootWalk ||
         current == GameAnims::EShootStrafe ) {
      return;
    }
  }
  
  // If we're shooting, we suggest that animation.
  if ( pC->shoot() ) {
    requested = GameAnims::EShoot;
  }
  

  // If we're walking, we suggest the walking animation.
  if ( pC->forward() || pC->backward() ) {
    requested = GameAnims::EWalk;
    // But if we're shooting at the same time, we suggest the shooting-walking
    // animation.
    if ( pC->shoot() ) {
      // We're walking and shooting
      requested = GameAnims::EShootWalk;
    }
    
  } else if ( pC->left() || pC->right() ) {
    // We're strafing
    requested = GameAnims::EStrafe;
    if ( pC->shoot() ) {
      // We're strafing and shooting
      requested = GameAnims::EShootStrafe;
    }
  }
  
  // Change the animation if new suggested animation is different from
  // the current animation. We also set up the new animation if player
  // just changed the weapon.
  if ( (requested != current) || (pC->weaponChange() != 0) ) {
    const Animation& anim = GameAnims::getAnimByWeapon( m_weapon.id(), requested );
    this->setAnimation( anim, requested );
  }
  
  // Reset the weaponChange-flag.
  pC->weaponChange() == 0;
}




/** Changes the weapon if requested weapon is available.
 */
void Player::changeWeapon(int w)
{
  if ( w == m_weapon.id() ) {
    return;
  }
  if ( m_weaponList.at(w) == -1 ) {
    // Player don't have the requested weapon
    return;
  }

  // Turn the sniper mode off.
  if ( m_sniperMode == true ) {
    m_sniperMode = false;
  }
  // Change the weapon.
  m_weapon = Weapon( w );
  if ( w == Weapon::W_CROWBAR ) {
    Sound::playSample( SMP_SLASH, false );
  } else {
    Sound::playSample( SMP_NEWWEAPON, false );
  }
  
  this->setCounter( RELOAD_COUNTER_INDEX, WEAPON_CHANGE_DELAY );
  
  // We change the animation based on our new weapon
  int t = GameAnims::EIdle;
  const Animation& anim = GameAnims::getAnimByWeapon( m_weapon.id(), t );
  this->setAnimation( anim, t );
}



/** Moves player up
 */
void Player::moveUp()
{
  int a = this->angle();
  Vec2D moveVec;
  int animID = GameAnims::EWalk;
  if ( a >= 224 || a <= 32 ) {
    moveVec = forwardSpeed;
    
  } else if ( a >= 96 && a <= 160 ) {
    moveVec = backwardSpeed;

  } else {
    moveVec = strafeSpeed;
    animID = GameAnims::EStrafe;
    
  }
  
  this->moveAndSetAnim( moveVec, animID );
}



/** Moves player upright
 */
void Player::moveUpRight()
{
  int a = this->angle();
  Vec2D moveVec;
  int animID = GameAnims::EWalk;
  if ( a <= 64 ) {
    moveVec = forwardSpeed;

  } else if ( a >= 128 && a <= 192 ) {
    moveVec = backwardSpeed;

  } else {
    moveVec = strafeSpeed;
    animID = GameAnims::EStrafe;

  }

  moveVec.rotate(32);
  this->moveAndSetAnim( moveVec, animID );
}



/** Moves player right
 */
void Player::moveRight()
{
  int a = this->angle();
  Vec2D moveVec;
  int animID = GameAnims::EWalk;
  if ( a >= 32 && a <= 96 ) {
    moveVec = forwardSpeed;
    
  } else if ( a >= 160 && a <= 224 ) {
    moveVec = backwardSpeed;
    
  } else {
    moveVec = strafeSpeed;
    animID = GameAnims::EStrafe;
  }
  moveVec.rotate( 64 );
  this->moveAndSetAnim( moveVec, animID );
}



/** Moves player downright
 */
void Player::moveDownRight()
{
  int a = this->angle();
  Vec2D moveVec;
  int animID = GameAnims::EWalk;
  if ( a >= 64 && a <= 128 ) {
    moveVec = forwardSpeed;

  } else if ( a >= 192 ) {
    moveVec = backwardSpeed;

  } else {
    moveVec = strafeSpeed;
    animID = GameAnims::EStrafe;
  }
  moveVec.rotate( 96 );
  this->moveAndSetAnim( moveVec, animID );
}



/** Moves player down
 */
void Player::moveDown()
{
  int a = this->angle();
  Vec2D moveVec;
  int animID = GameAnims::EWalk;
  if ( a >= 96 && a <= 160 ) {
    moveVec = forwardSpeed;

  } else if ( a >= 224 || a <= 32 ) {
    moveVec = backwardSpeed;

  } else {
    moveVec = strafeSpeed;
    animID = GameAnims::EStrafe;
  }
  moveVec.rotate( 128 );
  this->moveAndSetAnim( moveVec, animID );
}



/** Moves player downleft
 */
void Player::moveDownLeft()
{
  int a = this->angle();
  Vec2D moveVec;
  int animID = GameAnims::EWalk;
  if ( a >= 128 && a <= 192 ) {
    moveVec = forwardSpeed;

  } else if (  a <= 64 ) {
    moveVec = backwardSpeed;

  } else {
    moveVec = strafeSpeed;
    animID = GameAnims::EStrafe;
  }
  moveVec.rotate( 160 );
  this->moveAndSetAnim( moveVec, animID );
}



/** Moves player left
 */
void Player::moveLeft()
{
  int a = this->angle();
  Vec2D moveVec;
  int animID = GameAnims::EWalk;
  if ( a >= 160 && a <= 224 ) {
    moveVec = forwardSpeed;

  } else if ( a >= 32 && a <= 96 ) {
    moveVec = backwardSpeed;

  } else {
    moveVec = strafeSpeed;
    animID = GameAnims::EStrafe;
  }
  moveVec.rotate( 192 );
  this->moveAndSetAnim( moveVec, animID );
}



/** Moves player upleft
 */
void Player::moveUpLeft()
{
  int a = this->angle();
  Vec2D moveVec;
  int animID = GameAnims::EWalk;
  if ( a >= 192 ) {
    moveVec = forwardSpeed;

  } else if ( a >= 64 && a <= 160 ) {
    moveVec = backwardSpeed;

  } else {
    moveVec = strafeSpeed;
    animID = GameAnims::EStrafe;
  }
  moveVec.rotate( 224 );
  this->moveAndSetAnim( moveVec, animID );
}



/** Moves the player and sets the animation
 */
void Player::moveAndSetAnim(const Vec2D& rM, int animID)
{
  PlayerController* pC = dynamic_cast<PlayerController*>( this->getController() );
  if ( animID == GameAnims::EWalk && pC->shoot() != 0  ) {
    animID = GameAnims::EShootWalk;
  } else if ( animID == GameAnims::EStrafe && pC->shoot() != 0 ) {
    animID = GameAnims::EShootStrafe;
  }
      
  this->move( rM );
  if ( this->getAnimID() != GameAnims::EThrowing ) {
    if ( this->getAnimID() != animID ) {
      const Animation& an = GameAnims::getAnimByWeapon( m_weapon.id(), animID );
      this->setAnimation(an, animID);
    }
  }
}



/** Checks and opens the doors
 */
void Player::checkDoors()
{
  for (int i=0; i < WarGlobals::doorList.size(); i++) {
    float dx = WarGlobals::doorList.at(i)->getMidX();
    float dy = WarGlobals::doorList.at(i)->getMidY();

    Vec2D distV = Vec2D(dx,dy) - this->position();

    if ( distV.length() < 60 ) {
      // We've found the door that player touched
      WarGlobals::doorList.at(i)->open();
      return;
    }
  }
}



/** A helpper method for mapping the weaponchange keys to correct WeaponID
 * values.
 */
Weapon::WeaponID Player::getSelectedWeaponId( int aKeyValue ) const
{
  switch ( aKeyValue ) {
    case ( 1 ): return Weapon::W_CROWBAR;
    case ( 2 ): return Weapon::W_RIFLE;
    case ( 3 ): return Weapon::W_SHOTGUN;
    case ( 4 ): return Weapon::W_FLAMETHROWER;
    case ( 5 ): return Weapon::W_MINIGUN;
    case ( 6 ): return Weapon::W_UZI;
    case ( 7 ): return Weapon::W_SNIPERRIFLE;
    case ( 8 ): return Weapon::W_ROCKETLAUNCHER;
    default: {
      break;
    }
  }
  LOG_MESSAGE( "W_NOWEAPON" );
  return Weapon::W_NOWEAPON;
}



/** Selects next weapon
 */
void Player::selectNextWeapon()
{
}



/** Selects the previous weapon
 */
void Player::selectPreviousWeapon()
{
}

void Player::getShieldValues( float& aShield, float& aMaxShield ) const
{
  aShield = m_shield;
  aMaxShield = Settings::floatObjProp( ObjectID::TYPE_PLAYER, "max_shield:" );
}

} // end of namespace
